package editor;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.util.ArrayList;

import javax.imageio.ImageIO;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

public class CharacterList extends JPanel implements ActionListener  {

	private ArrayList<CharacterListItem> listItems;
	
	private JPanel addPanel;
	
	private ImageIcon iconForNoImage;
	
	private boolean removePopupWasCancelled = true;
	
	public CharacterList() {
		
		this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
		
		listItems = new ArrayList<CharacterListItem>();
		
		addPanel = new JPanel(new FlowLayout());
		JButton addButton = new JButton("Add Character...");
		addButton.addActionListener(this);
		addPanel.add(addButton);
		
		JButton removeButton = new JButton("Remove Character...");
		removeButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent arg0) {
				removeCharacter();
				
			}
		});
		addPanel.add(removeButton);
		
		this.add(addPanel);
		
		thoroughlySetDimensions(new Dimension(275, 100));
		
		
		try {
			BufferedImage im = ImageIO.read(CharacterList.class.getResource("/images/no_image_red_white.GIF"));
			iconForNoImage = new ImageIcon(im);
		} catch(Exception e) {
			JOptionPane.showMessageDialog(null, "Error while trying to load the 'no image' icon");
		}
		
		
		
	}
	
	
	private void thoroughlySetDimensions(Dimension d) {
		
		this.setSize(d);
		this.setPreferredSize(d);
		this.setMinimumSize(d);
		this.setMaximumSize(d);
		
	}


	@Override
	public void actionPerformed(ActionEvent arg0) {
		
		char newChar;
		
		while(true) {
		
			newChar = CharacterListItem.popUpCharacterSelectDialog('\0');
			
			if(newChar == '\0')
				return;
			else if(activationKeyTaken(newChar)) {
				int rety = JOptionPane.showConfirmDialog(null, "The key you entered, " + Character.toString(newChar) + ", was already taken.  Try another?","Error", JOptionPane.YES_NO_OPTION);
				if(rety == JOptionPane.NO_OPTION)
					return;
			} else {
				break;
			}
			
		}
		
		
		JDialog frame = new JDialog();
		
		PlayerCharacterEditorPanel panel = new PlayerCharacterEditorPanel(frame);
		
		frame.setContentPane(panel);
		
		frame.setPreferredSize(new Dimension(500, 600));
		frame.setMinimumSize(new Dimension(500, 600));
		frame.setMaximumSize(new Dimension(500, 600));
		
		frame.setModal(true);
		
		frame.setVisible(true);
		
		
		if(panel.wasCancelled() || panel.getContentArray() == null) {
			System.out.println("<><>DELME either it was cancelled or null:  cancelled? " + panel.wasCancelled() + ", null?  " + (panel.getContentArray() == null));
			return;
		}
			
		
		String contentArray[] = panel.getContentArray();
		
		CharacterListItem cli = new CharacterListItem(contentArray, newChar, iconForNoImage, this);
		
		
		this.remove(addPanel);
		this.add(cli);
		this.add(addPanel);
		
		this.listItems.add(cli);
		
		Dimension origD = this.getPreferredSize();
		
		Dimension newD = new Dimension(origD.width, origD.height + 100);
		
		this.thoroughlySetDimensions(newD);
		
		this.revalidate();
		this.repaint();
		
	}
	
	
	public boolean activationKeyTaken(char newCharacter) {
		
		if(listItems == null || listItems.size() == 0) {
			return false;
		}
		
		for(CharacterListItem cli : listItems) {
			if(Character.toUpperCase(cli.getActivateKey()) == Character.toUpperCase(newCharacter))
				return true;
		}
		
		return false;
		
	}
	
	
	public void removeCharacter() {
		
		removePopupWasCancelled = true;
		
		JPanel popupPanel = new JPanel(new BorderLayout());
		DefaultComboBoxModel listModel = new DefaultComboBoxModel();
		JComboBox comboBox = new JComboBox(listModel);
		
		System.out.println("<><>DELME listItems: " + this.listItems.size());
		
		for(CharacterListItem cli : this.listItems) {
			System.out.println("<><>DELME adding this to the list: " + cli);
			listModel.addElement(cli);
		}
		
		
		JPanel mainPanel = new JPanel(new FlowLayout());
		mainPanel.add(new JLabel("Select character to be deleted:"));
		mainPanel.add(comboBox);
		
		popupPanel.add(mainPanel, BorderLayout.CENTER);
		
		JPanel bottomPanel = new JPanel(new GridLayout(1, 4));
		
		final JDialog frame = new JDialog();
		
		JButton cancel = new JButton("Cancel");
		cancel.addActionListener(new ActionListener() {
			
			@Override
			public void actionPerformed(ActionEvent arg0) {
				frame.dispose();
			}
		});
		
		bottomPanel.add(cancel);
		
		bottomPanel.add(new JPanel());
		bottomPanel.add(new JPanel());
		
		JButton ok = new JButton("OK");
		ok.addActionListener(new ActionListener() {
			
			@Override
			public void actionPerformed(ActionEvent arg0) {
				removePopupWasCancelled = false;
				frame.dispose();
			}
		});
		bottomPanel.add(ok);
		
		popupPanel.add(bottomPanel, BorderLayout.SOUTH);
		
		frame.setContentPane(popupPanel);
		
		Dimension preferredD = new Dimension(400, 300);
		frame.setSize(preferredD);
		frame.setPreferredSize(preferredD);
		frame.setModal(true);
		
		frame.setVisible(true);
		
		
		frame.dispose();
		
		Object o = comboBox.getSelectedItem();
		
		if(o == null || removePopupWasCancelled || !(o instanceof CharacterListItem))
			return;
		
		
		
		CharacterListItem toBeErased = (CharacterListItem) o;
		
		this.remove(toBeErased);
		
		this.listItems.remove(comboBox.getSelectedIndex());
		
		this.revalidate();
		this.repaint();
	}
	
	
	
	public void loadCharactersFromFile(BufferedReader in) throws Exception {
		
		this.remove(addPanel);
		
		for(CharacterListItem cli : this.listItems) {
			this.remove(cli);
		}
		
		this.listItems.clear();
		
		this.listItems = new ArrayList<CharacterListItem>();
		
		
		String inputter;
		
		while((inputter = in.readLine()) != null) {
			
			if(inputter.contains("#"))
				continue;
			
			if(inputter.equals("/characters"))
				break;
			
			if(inputter.contains("character:")) {
				
				char activationKey = inputter.charAt(10);
				
				CharacterListItem newCli = CharacterListItem.readCharacterFromFile(in, activationKey, iconForNoImage, this);
				
				this.listItems.add(newCli);
				
				this.add(newCli);
			} else if(inputter.contains("charNoCompleta:")) {
				
				char activationKey = inputter.charAt(15);
				
				CharacterListItem newCli = CharacterListItem.readCharacterFromFile(in, activationKey, iconForNoImage, this);
				
				if(newCli == null)
					break;
				
				this.listItems.add(newCli);
				
				this.add(newCli);
			}
			
		}
		
		this.add(addPanel);
		
		Dimension newD = new Dimension(275, 100 * this.listItems.size() + 100);
		
		this.thoroughlySetDimensions(newD);
		
		
		this.revalidate();
		this.repaint();
		
	}
	
	public int getCharacterCount() {
		if(this.listItems == null)
			return -1;
		
		return this.listItems.size();
	}
	
	public ArrayList<CharacterListItem> getCharacters() {
		return this.listItems;
	}
	
	public void removeAllCharacters() {
		
		
		if(this.listItems != null) {
			for(CharacterListItem cli : listItems) {
				this.remove(cli);
			}
			this.listItems.clear();
		}
		this.listItems = new ArrayList<CharacterListItem>();
		
		
		this.thoroughlySetDimensions(addPanel.getSize());
		
	}
}
